//
// Copyright 2015-2021 by Garmin Ltd. or its subsidiaries.
// Subject to Garmin SDK License Agreement and Wearables
// Application Developer Agreement.
//

import Toybox.Communications;
import Toybox.Cryptography;
import Toybox.Lang;

class Huum {

private const HUUM_NAME     = "Sauna";

private const STATE_UNKNOWN     =   0;
private const STATE_OFFLINE     = 230;
private const STATE_HEATING     = 231;
private const STATE_COOLING     = 232;
private const STATE_LOCKED      = 233;
private const STATE_STOPED      = 400;

private var http;

private var cb;
private var dev as Dictionary or Null;

public function initialize()
{

    L.dbg_log(L.DBG_BASIC, "Huum:initialize()");
    http = new Http(C.settings["huum_svr"]);
    var token = C.b64_encode(C.settings["huum_usr"] + ":" + C.settings["huum_pwd"]);
    http.set_opts({"Content-Type" => Communications.REQUEST_CONTENT_TYPE_URL_ENCODED,
                   "Connection" => "close",
                   "Authorization" => "Basic " + token },
                  Communications.HTTP_RESPONSE_CONTENT_TYPE_JSON);
}

public function avail()
{

    return !C.empty(C.settings["huum_svr"]) && !C.empty(C.settings["huum_usr"]);
}

public function get_state(cb)
{


    http.get("/action/home/status", null, cb, method(:get_state_cb));
}

public function get_state_cb(code, data as Dictionary, cb)
{

    if (code != 200) {
        L.dbg_log(L.DBG_HTTP, "Huum get failed - " + code);
        cb.invoke(code, 0,  0);
        return;
    }
//L.pr_json(L.DBG_HTTP, "huum", data);
    var status = data["statusCode"];
    cb.invoke(status,
              data["temperature"].toNumber(),
              (status == 231) ? data["targetTemperature"].toNumber() : null);
}

private function status2state(status)
{

//    if (status < 0) {
//        return status; }

    switch (status) {

    case STATE_OFFLINE:
        return G.STATE_OFL;

    case STATE_HEATING:
        return G.STATE_ON;

    case STATE_COOLING:
        return G.STATE_OFF;

    case STATE_LOCKED:
        return G.STATE_OFL;

    case STATE_STOPED:
        return G.STATE_OFL;

    }
    return G.STATE_UNK;
}

private function enumerate()
{

    get_state(method(:enumerate_cb));
}

public function enumerate_cb(status, temp, t_temp)
{

    if (status > 0) {
        C.outlets.add({
                        "type" => C.OUTLET_HUUM,
                        "name" => HUUM_NAME,
                        "icon" => G.ICON_SAUNA,
                        "state" => status2state(status),
                        "ood" => false,
                        "info" => {
                                    "cur_temp" => temp,
                                    "target_temp" => t_temp
                                  }
                      });
    }
    cb.invoke(C.OUTLET_HUUM + 1, null);
}

private function huum_get(dev)
{

    me.dev = dev;
    get_state(method(:huum_get_cb));
}

public function huum_get_cb(status, temp, t_temp)
{

    if (status < 0) {
        cb.invoke(C.OUTLET_ERROR, C.err_msg(status));
        return;
    }

    if (status != STATE_UNKNOWN) {
        var info = dev["info"] as Dictionary;
        info["cur_temp"] = temp;
        info["target_temp"] = t_temp;
    }
    cb.invoke(status2state(status), null);
}

private function huum_set(state, dev)
{
    var cmd, body;

    me.dev = dev;

    if (state == G.STATE_ON) {
        cmd = "/action/home/start";
        body = {"targetTemperature" => C.settings["huum_temp"]};
    } else {
        cmd = "/action/home/stop";
        body = null;
    }
    http.post(cmd, body, cb, method(:huum_set_cb));
}

public function huum_set_cb(code, data as Dictionary, cb)
{

//L.pr_json(L.DBG_HTTP, "huum_set_cb:" + code, data);
    if (code != 200) {
        cb.invoke(C.OUTLET_ERROR, C.err_msg(code));
        return;
    }
    var status = data["statusCode"];
    if (status != STATE_UNKNOWN) {
        var info = dev["info"] as Dictionary;
        info["cur_temp"] = data["temperature"].toNumber();
        info["target_temp"] = ((status == 231) ? data["targetTemperature"].toNumber() : null);
    }
    cb.invoke(status2state(status), null);
}

public function ctrl(cmd, outlet as Dictionary, state, cb)
{

    me.cb = cb;

    switch (cmd) {

    case C.CTRL_ENUMERATE:
        enumerate();
        return;

    case C.CTRL_GET:
        huum_get(outlet);
        return;

    case C.CTRL_SET:
        huum_set(state, outlet);
        return;

    }
    L.dbg_log(L.DBG_BASIC, "ctrl:" + "unknown command" + cmd);
}

}
