using Toybox.Application.Properties;
using Toybox.Graphics as Gfx;
using Toybox.Math;
using Toybox.System as Sys;
using Toybox.Time;
using Toybox.Time.Gregorian;

import Toybox.Application.Storage;
import Toybox.Communications;
import Toybox.Lang;
import Toybox.StringUtil;
import Toybox.WatchUi;

module C
{

//
//  Define my own colors since Garmin is a little color
//  blind (0xff00ff is magenta, not pink).
//
public const BLACK   = 0x000000;
public const WHITE   = 0xffffff;
public const GRAY    = 0x888888;
public const RED     = 0xff0000;
public const GREEN   = 0x00ff00;
public const BLUE    = 0x0000ff;
public const CYAN    = 0x00ffff;
public const MAGENTA = 0xff00ff;
public const YELLOW  = 0xffff00;

//
//  Main view statews
//
public const MAIN_SCAN          = 0;
public const MAIN_WAIT          = 1;
public const MAIN_ENUMERATE     = 2;
public const MAIN_SHOW          = 3;
public const MAIN_ERROR         = 4;
public const MAIN_LOG           = 5;

//
//  Intelligent outlet types
//
public const OUTLET_TPLINK = 0;
public const OUTLET_SHELLY = 1;     // must be equal to OUTLET_TPLINK + 1
public const OUTLET_ATHOME = 2;     // must be equal to OUTLET_SHELLY + 1
public const OUTLET_HUUM   = 3;     // must be equal to OUTLET_ATHOME + 1
public const OUTLET_NONE   = 4;     // must be equal to OUTLET_HUUM + 1
public const OUTLET_ERROR  = 5;

//
//  Outlet controls
//
public const CTRL_GET       = 0;
public const CTRL_SET       = 1;
public const CTRL_ENUMERATE = 2;

//
//  icons
//

public var icons;
public const icons_med = [
    Rez.Drawables.outlet_off_med, Rez.Drawables.outlet_on_med, Rez.Drawables.outlet_ofl_med,
    Rez.Drawables.strip_off_med,  Rez.Drawables.strip_on_med,  Rez.Drawables.strip_ofl_med,
    Rez.Drawables.switch_off_med, Rez.Drawables.switch_on_med, Rez.Drawables.switch_ofl_med,
    Rez.Drawables.bulb_off_med,   Rez.Drawables.bulb_on_med,   Rez.Drawables.bulb_ofl_med,
    Rez.Drawables.sauna_off_med,  Rez.Drawables.sauna_on_med,  Rez.Drawables.sauna_ofl_med,
];
public const icons_lg = [
    Rez.Drawables.outlet_off_lg, Rez.Drawables.outlet_on_lg, Rez.Drawables.outlet_ofl_lg,
    Rez.Drawables.strip_off_lg,  Rez.Drawables.strip_on_lg,  Rez.Drawables.strip_ofl_lg,
    Rez.Drawables.switch_off_lg, Rez.Drawables.switch_on_lg, Rez.Drawables.switch_ofl_lg,
    Rez.Drawables.bulb_off_lg,   Rez.Drawables.bulb_on_lg,   Rez.Drawables.bulb_ofl_lg,
    Rez.Drawables.sauna_off_lg,  Rez.Drawables.sauna_on_lg,  Rez.Drawables.sauna_ofl_lg,
];

//
//  Menu constants
//
public const MENU_NONE      = 0;
public const MENU_ONOFF     = 1;
public const MENU_DEVICE    = 100;

//
//  Watch face dimensions
//
public var cx;
public var cy;
public var radius;

//
//  Common data
//
public var state = MAIN_SCAN;
public var alert_hdr = "Wait internet";
public var alert_body = null;
public var outlets as Array<Dictionary> = new [0];
public var dev_list;
public var enumerating = OUTLET_TPLINK;
public var onoff_state;
public var onoff_ood;

public var tplink;
public var shelly;
public var athome;
public var huum;

//
//  Settings
//
public var settings as Dictionary<Object, String or Number or Boolean or Array<String>> = {
    "offline"       =>  false,
    "sort_dir"      =>  1,      // -1 == descending, 0 == none, 1 = ascending
    "temp_unit"     =>  0,      // 0 == fahrenheit, 1 == celsius
    "tp_svr"        =>  "",
    "tp_usr"        =>  "",
    "tp_pwd"        =>  "",
    "shelly_svr"    =>  "",
    "shelly_key"    =>  "",
    "athome_svr"    =>  "",
    "athome_usr"    =>  "",
    "athome_pwd"    =>  "",
    "huum_svr"      =>  "",
    "huum_usr"      =>  "",
    "huum_pwd"      =>  "",
    "huum_temp"     =>  0,
};

function get_settings()
{

    P.set_prop("version", L.version);

    settings["offline"] = P.get_prop("offline", true);
    var unit = P.get_prop("temp_unit", 0);
    settings["temp_unit"] = unit;

    switch (P.get_prop("sort_dir", 1)) {

    case 0:     // descending
        settings["sort_dir"] = -1;
        break;

    case 1:     // ascending
        settings["sort_dir"] = 1;
        break;

    default:    // no sort at all
        settings["sort_dir"] = 0;
        break;

    }

    settings["tp_svr"] = P.get_prop("tp_svr", "");
    settings["tp_usr"] = P.get_prop("tp_usr", "");
    settings["tp_pwd"] = P.get_prop("tp_pwd", "");

    settings["shelly_svr"] = P.get_prop("shelly_svr", "");
    settings["shelly_key"] = P.get_prop("shelly_key", "");

    settings["athome_svr"] = P.get_prop("athome_svr", "");
    settings["athome_usr"] = P.get_prop("athome_usr", "");
    settings["athome_pwd"] = P.get_prop("athome_pwd", "");

    settings["huum_svr"] = P.get_prop("huum_svr", "");
    settings["huum_usr"] = P.get_prop("huum_usr", "");
    settings["huum_pwd"] = P.get_prop("huum_pwd", "");
    settings["huum_temp"] = celsius(P.get_prop("huum_temp", ((unit == 0) ? 160 : 74)));

    if (L.debug != L.DBG_NONE) {
        settings["offline"] = true;
        settings["temp_unit"] = 0;
        settings["tp_svr"] = "https://wap.tplinkcloud.com";
        settings["tp_usr"] = "n0ano@n0ano.com";
        settings["shelly_svr"] = "https://shelly-160-eu.shelly.cloud";
        settings["athome_svr"] = "n0ano.com/cgi-bin/garmin.pl";
settings["athome_svr"] = "";
        settings["athome_usr"] = "n0ano";
        settings["huum_svr"] = "https://api.huum.eu";
        settings["huum_usr"] = "n0ano@n0ano.com";
        settings["huum_temp"] = 74;
    }

    L.pr_json(L.DBG_BASIC, "settings " + L.version + ":", settings);
}

//
//  Garmin doesn't provide an API to generate a UUID
//  and the UUID provided to a TpLink API doesn't have
//  be that unique.  To simplify things we'll use a
//  template from a working UUID and just provide make
//  the first 32 bits randome, should be sufficient.
function fake_uuid()
{
    var r;

    Math.srand(Time.now().value());
    r = Math.rand();
    r = r.format("%8.8x") + "-a5b5-4774-86e6-15338b4f0149";
    return r;
}

function set_fill(dc, color)
{

    if (dc has :setFill) {
        dc.setFill(color);
    } else {
        dc.setColor(color, Gfx.COLOR_TRANSPARENT);
    }
}

function set_alert(hdr, body)
{

    if (hdr == null && body == null) {
        hdr = null;
        body = null;
        return;
    }

    if (hdr != null) {
        alert_hdr = hdr; }
    if (body != null) {
        alert_body = body; }
    WatchUi.requestUpdate();
}

//
//  Note that the callback for these APIs includes 2 paramters,
//  the first is the return value and the second is either a
//  null (the return value is valid) or an error string (the
//  return value is arbitrary).
function dev_ctrl(op, dev as Dictionary, state, cb)
{

    var type = dev["type"];
    switch (type) {

    case OUTLET_TPLINK:
        tplink.ctrl(op, dev, state, cb);
        return;

    case OUTLET_SHELLY:
        shelly.ctrl(op, dev, state, cb);
        return;

    case OUTLET_ATHOME:
        athome.ctrl(op, dev, state, cb);
        return;

    case OUTLET_HUUM:
        huum.ctrl(op, dev, state, cb);
        return;

    }
    L.dbg_log(L.DBG_BASIC, "dev_ctrl:" + type + " => unknown device type");
    cb.invoke(false);
}

function sh_progress(msg, delegate)
{

    var pb = new WatchUi.ProgressBar(msg, null);
    WatchUi.pushView(pb, delegate, WatchUi.SLIDE_IMMEDIATE);
}

function load_devs()
{
    var dev;

    outlets = Storage.getValue("Devices");
    if (outlets != null) {
        for (var i = 0; i < C.outlets.size(); i += 1) {
            dev = C.outlets[i];
            dev["ood"] = true;
        }
        return true;
    }
    return false;
}

function save_devs(dev)
{

    Storage.setValue("Devices", outlets);
    if (dev != null) {
        Storage.setValue("Glance_dev", dev); }
}

function state2str(s)
{

    switch (s) {

    case G.STATE_OFL:
        return "offline";

    case G.STATE_OFF:
        return "off";

    case G.STATE_ON:
        return "on";

    }

    return "??";
}

function outlet_sort(dir)
{
    var dev;

    for (var i = 0; i < outlets.size() - 1; i += 1) {
        for (var j = i + 1; j < outlets.size(); j += 1) {
            if (dev_comp(outlets[i], outlets[j], dir) < 0) {
                dev = outlets[i];
                outlets[i] = outlets[j];
                outlets[j] = dev;
            }
        }
    }
}

//
// 66 characters, the last one ('=') is not part of an encoding, it is
// tacked on to the end to make it easier to idenfy in a source string
// (Monkey C makes dealing with strings/chars/ints `interesting`.
//
const b64_table = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
function b64_decode(str)
{

    var bits = 0;
    var ac = 0;
    var r = "";

    for (var i = 0; i < str.length(); i += 1) {
        var c = str.substring(i, i + 1);
        var idx = b64_table.find(c);
        if (idx == null) {
            return null; }
        if (idx == 64) {     // this char is an '='
            ac = ac << 6;
            bits += 6;
            if (bits >= 8) {
                bits -= 8; }
        } else {
            ac = (ac << 6) | idx;
            bits += 6;
            if (bits >= 8) {
                bits -= 8;
                var cc = ((ac >> bits) & 0xff).toChar().toString();
                r += cc;
            }
        }
    }
    if ((ac & ((1 << bits) - 1)) != 0) {
        return null; }
    return r;
}

function b64_encode(src)
{   
    var idx;
    var ac = 0;
    var bits = 0;
    var dst = "";

    var b64_chars = src.toCharArray() as Array<Char>;
    for (var i = 0; i < b64_chars.size(); i += 1) {
        idx = b64_chars[i].toNumber();
        ac = (ac << 8) | idx;
        bits += 8;
        do {
            bits -= 6;
            idx = (ac >> bits) & 0x3f;
            dst += b64_table.substring(idx, idx + 1);
        } while (bits >= 6);
    }
    if (bits) {
        idx = (ac << (6 - bits)) & 0x3f;
        dst += b64_table.substring(idx, idx + 1);
        bits -= 6;
    }   
    while (bits < 0) {
        dst += "=";
        bits += 2;
    }
    return dst;
}

function c2f(c)
{

    if (c == null) {
        return null; }
    return ((c * (9.0/5.0)) + 32.0).toNumber();
}

function f2c(f)
{

    if (f == null) {
        return null; }
    return ((f - 32.0) * (5.0/9.0)).toNumber();
}

function celsius(t)
{

    if ((t <= 0) || (settings["temp_unit"] == 1)) {
        return t; }
    return f2c(t);
}

function err_msg(code)
{

    return (code == Communications.BLE_CONNECTION_UNAVAILABLE) ? "No phone" : ("Error:" + code);
}

function empty(str)
{

    return (str == null) || (str.length() == 0);
}

function dev_comp(a as Dictionary, b as Dictionary, dir)
{

    var n1 = a["name"];
    var n2 = b["name"];
    return compare_to(n1, n2) * dir;
}

function compare_to(s1, s2)
{
    var n;

    var b1 = s1.toUtf8Array() as Array<Number>;
    var l1 = b1.size();
    var b2 = s2.toUtf8Array() as Array<Number>;
    var l2 = b2.size();
    var len = (l1 < l2) ? l1 : l2;
    for (var i = 0; i < len; i += 1) {
        n = b2[i] - b1[i];
        if (n != 0) {
            return  n; }
    }
    if (l1 == l2) {
        return 0;
    } else if (l1 < l2) {
        return 1;
    } else {
        return -1; }
}

}
